package gov.vha.vuid.rest.data;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.text.NumberFormat;
import java.util.Enumeration;
import java.util.Optional;
import java.util.Properties;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.glassfish.hk2.runlevel.RunLevel;
import org.jvnet.hk2.annotations.Service;
import gov.vha.isaac.ochre.api.LookupService;
import gov.vha.isaac.rest.api.data.vuid.RestVuidBlockData;
import gov.vha.isaac.rest.api.exceptions.RestException;
import gov.vha.vuid.rest.ApplicationConfig;

@Service(name="vuid-service")
@RunLevel(LookupService.METADATA_STORE_STARTED_RUNLEVEL)
public class VuidServiceImpl implements VuidService
{
	private static final Logger LOG = LogManager.getLogger(VuidService.class);
	private static final String CONFIG_RESOURCE = "vuid-rest.properties";
	private Properties PROPS = new Properties();
	
	private DatabaseType databaseType_ = DatabaseType.H2; // Default
	private Connection dbConnection_;
	private boolean prepared_ = false;
	private boolean useRealVuids_ = false;
	private File propsBakFile_;
	
	// Defines the type of database in use
	private static enum DatabaseType
	{
		H2, H2_MEM, ORACLE, NONE;
	}
	
	/*
	 * Standard constructor, no default actions. - for HK2 only
	 */
	private VuidServiceImpl()
	{
		
	}
	
	@PostConstruct
	private void prepareDatabase()
	{
		try {
			LOG.debug("Prepare DB starting");
			ApplicationConfig.getInstance().setStatus("Preparing Environment.");

			loadProperties();
			if (ApplicationConfig.getInstance().isShutdownRequested())
			{
				return;
			}

			determineDatabaseType();
			if (ApplicationConfig.getInstance().isShutdownRequested())
			{
				return;
			}
			setup();

			LOG.debug("Database type: " + this.databaseType_);
			LOG.debug("Connection ready: " + this.isReady());
			LOG.debug("DB test result: " + this.testDatabase());
			LOG.info("Prod DB : " + this.isProductionMode());

			ApplicationConfig.getInstance().setStatus("Startup Complete" + (this.isReady() ? "" : " But failed!"));
		} catch (RuntimeException re) {
			LOG.error(re.getClass().getSimpleName() + " thrown in prepareDatabase(): \"" + re.getMessage() + "\"", re);
			throw re;
		}
	}
	
	
	/**
	 * Load all the properties files needed to configure/prepare the database.
	 */
	private void loadProperties()
	{
		// Load default properties
		PROPS.putAll(loadDefaultProperties());
		
		// Load overrides
		PROPS.putAll(loadCustomProperties());
		
		configureDbPath();
		propsBakFile_ = new File(PROPS.getProperty("vuid.data.path", PROPS.getProperty("default.vuid.data.path")), CONFIG_RESOURCE + ".bak");
		
		// Load saved properties from previous run
		PROPS.putAll(loadSavedProperties());
		
		String dbConfigPath = PROPS.getProperty("prisme.data.path", PROPS.getProperty("default.prisme.data.path"));
		String dbConfigYaml = PROPS.getProperty("prisme.db.config", PROPS.getProperty("default.prisme.db.config"));
		
		// Load database_vuid.yml via File
		File dbConfigFile = new File(dbConfigPath, dbConfigYaml);

		PROPS.putAll(loadPropertiesInWindow(dbConfigFile));

		LOG.info("Finished loading properties.");
		LOG.debug(getSanitizedProperties(PROPS));	
	}
	
	
	/**
	 * Returns a Properties object which contains the default
	 * configuration values to be used, if needed.
	 * 
	 * @return A Properties object of the default properties
	 */
	private Properties loadDefaultProperties()
	{
		Properties p_new = new Properties();

		LOG.debug("Loading " + "/" + CONFIG_RESOURCE + " as project resource...");
		// Load CONFIG_RESOURCE (vuid-rest.properties) as project resource
		try (InputStream stream = VuidServiceImpl.class.getResourceAsStream("/" + CONFIG_RESOURCE))
		{
			p_new.load(stream);
			p_new.putAll(getPrependedProperties(p_new, "default"));
			LOG.debug("Loaded " + "/" + CONFIG_RESOURCE + " as project resource.");
		}
		catch (Exception e)
		{
			String msg = "Terminal failure. Unexpected error accessing or loading default Properties file " + "/" + CONFIG_RESOURCE + " as project resource";
			LOG.error(msg, e);
			throw new RuntimeException(msg, e);
		}
		
		// Check that the given data dir is accessible, otherwise write
		// to the Java tmpdir location
		boolean useJavaTmpDir = false;
		
		try
		{
			File f = new File(p_new.getProperty("default.vuid.data.path"));
			if (!f.exists() || !f.canWrite())
			{
				useJavaTmpDir = true;
			}
		}
		catch (NullPointerException npe)
		{
			useJavaTmpDir = true;
		}
		
		if (useJavaTmpDir)
		{
			LOG.error("Cannot access VUID Server data directory set as: "
					+ p_new.getProperty("vuid.data.path")
					+ ", defaulting to: " + System.getProperty("java.io.tmpdir")
					+ " for operation.");
			p_new.put("default.vuid.data.path", System.getProperty("java.io.tmpdir"));
		}
		
		LOG.info("Default properties loaded.");
		LOG.debug(getSanitizedProperties(p_new));
		
		return p_new;
	}
	
	
	/**
	 * This will store the properties values of the current running
	 * configuration. On startup, these values are reloaded. If we
	 * see an old Prisme database_vuid.yml file, we can compare those
	 * values to the values we saved to see if we are using a current
	 * file. This is an attempt to work around the scenario where
	 * Prisme was started long before the VUID-rest application and
	 * we won't have a fresh database_vuid.yml file to load.  
	 * 
	 * @param props The properties object to store to disk
	 * @return A Properties object of the saved properties
	 */
	private Properties saveProperties(Properties props) throws IOException
	{
		Properties p = new Properties();
		
		// We don't want to persist Oracle connection information
		//p.putAll(getSanitizedProperties(props));
		// Currently, the only important value is the generated time
		//p.put("saved.epoch_time_seconds", props.getProperty("epoch_time_seconds", "0"));
		p = getPrependedProperties(props, "saved");
		p = getSanitizedProperties(p);

		FileOutputStream fos = new FileOutputStream(propsBakFile_);
		p.store(fos, "VUID-Rest Properties Backup");
		fos.close();
		
		return p;
	}
	
	
	/**
	 * This will load the previously stored properties of the last
	 * known good Prisme configuration.
	 * 
	 * @return A Properties object of the previously saved properties
	 */
	private Properties loadSavedProperties()
	{
		Properties p = new Properties();
		
		// Load saved properties from previous run
		try (InputStream stream = new FileInputStream(propsBakFile_))
		{
			p.load(stream);
			LOG.info("Saved properties loaded.");
			LOG.debug(getSanitizedProperties(p));
		}
		catch (Exception e)
		{
			String msg = "Exception encountered when trying to load the backup properties file: " + propsBakFile_.getAbsolutePath()
			+ System.getProperty("line.separator") 
			+ e.getClass().getSimpleName() + ": " + e.getMessage();
			LOG.info(msg);
		}
		
		return p;
	}
	
	
	/**
	 * Load custom Property overrides.
	 * 
	 * @return A Properties object of the custom property overrides
	 */
	private Properties loadCustomProperties()
	{
		Properties p = new Properties();
		
		File f = new File(PROPS.getProperty("vuid.data.path"), "custom-" + CONFIG_RESOURCE);
		try (InputStream stream = new FileInputStream(f))
		{
			if (f.exists())
			{
				;
				p.load(stream);
				LOG.info("Custom properties loaded.");
			}
			else
			{
				LOG.info("No custom properties found.");
			}
		} 
		catch (FileNotFoundException fnfe) 
		{
			LOG.info("No custom properties found.");
		} 
		catch (IOException ioe)
		{
			String msg = "Exception encountered when trying to load the custom properties file."
			+ System.getProperty("line.separator") 
			+ ioe.getClass().getSimpleName() + ": " + ioe.getMessage();
			LOG.error(msg);
		}
		
		return p;
	}
	
	
	/**
	 * This method attempts to load the database_vuid.yml file that is
	 * created by Prisme and found in the prisme.data.path directory.
	 * This method blocks the thread for up to THREAD_MAX_SLEEP_MILLIS,
	 * but should only do it once, on application startup. It should
	 * never return null.
	 * 
	 * It covers a number of edge cases, but tries to follow this process:
	 *  
	 *  - If the file exists, reads the file and checks that the 
	 *	'epoch_time_seconds' value is <= FILE_MAX_TIME_DIFF_MILLIS
	 *  	~ Is so, the file is read in and the values returned 
	 *  	~ If not, it waits for THREAD_SLEEP_INTERVAL_MILLIS and
	 *  	  checks again, waiting for a new file. 
	 *  		^ If a new file is found, it's read in and the values returned
	 *  		^ If a new file is not found before THREAD_MAX_SLEEP_MILLIS
	 *  		  is reached, it gives up waiting.
	 *  
	 *  - If a file is not found, or is designated as 'too old' (i.e. the
	 *	'epoch_time_seconds' value is greater than FILE_MAX_TIME_DIFF_MILLIS
	 *	from System.currentTimeMillis()), then:
	 *  	~ The saved properties from the previously known good 
	 *  	  configuration values are checked. 
	 *  		^ If the 'epoch_time_seconds' values match, the file given is 
	 *  		  used with the assumption that Prisme hasn't restarted alongside 
	 *  		  the VUID-rest service.
	 *			^ If the two 'epoch_time_seconds' values differ, the application
	 *			  assumes there is a problem and goes into non-Production mode,
	 *			  which will try configuring a H2 database on disk, then in memory.
	 * 
	 * If it is determined that this is deployed in 'debug mode', the effective sleep
	 * time above is 0.  debug mode occurs when the war file is a SNAPSHOT, or, when
	 * it is running directly from eclipse. 
	 *
	 * @param file The File object of the Prisme database_vuid.yml file
	 * @return A Properties object, possibly of the database_vuid.yml contents
	 */
	private Properties loadPropertiesInWindow(File file)
	{
		// These are set via the CONFIG_RESOURCE or custom properties file
		long fileMaxTimeDiffMillis = Long.parseLong(
				PROPS.getProperty("vuid.file.max_time_diff_millis", PROPS.getProperty("default.vuid.file.max_time_diff_millis", "1")));
		long threadMaxSleepMillis = Long.parseLong(
				PROPS.getProperty("vuid.thread.max_sleep_millis", PROPS.getProperty("default.vuid.thread.max_sleep_millis", "1")));
		
		LOG.info("Preparing to load Prisme config file " + file.getAbsolutePath() + " within window of " + (fileMaxTimeDiffMillis / 1000) + " seconds.");
		
		Properties p = new Properties();
		
		String currPropTimeVal = "0";
		
		InputStream stream = null;
		
		ApplicationConfig.getInstance().setStatus("Loading Prisme Config.");
		
		long loopUntil = threadMaxSleepMillis + System.currentTimeMillis();
		
		try 
		{
			while (!ApplicationConfig.getInstance().isShutdownRequested() && System.currentTimeMillis() < loopUntil)
			{
				if (file != null && file.exists())
				{
					LOG.debug("Reading " + file.getAbsolutePath());
					stream = new FileInputStream(file);
					p.load(stream);
					currPropTimeVal = p.getProperty("epoch_time_seconds");
					long currPropTime = currPropTimeVal == null ? 0L : Long.valueOf(currPropTimeVal).longValue() * 1000;
					
					if (currPropTime >= System.currentTimeMillis() // ?? Could it happen?
						|| (System.currentTimeMillis() - currPropTime) <= fileMaxTimeDiffMillis)
					{
						LOG.info("Saving configuration entries.");
						saveProperties(p);
						LOG.info("Leaving wait loop");
						break;
					}
				}
				
				if (ApplicationConfig.getInstance().isDebugDeploy())
				{
					//Prisme isn't likely going to show up.  No reason to wait.
					LOG.info("DEBUG DEPLOY DETECTED, skipping sleep - should proceed to non-prod mode");
					break;
				}
				
				//check for the file once per second
				long sleepFor = Math.min(1000, (loopUntil - System.currentTimeMillis()));
				if (sleepFor > 0)
				{
					try
					{
						LOG.trace("Waiting for config file to appear");
						synchronized (LOG)
						{
							LOG.wait(sleepFor);
						}
					}
					catch (InterruptedException e)
					{
						//noop
					}
				}
			}
			
			if (ApplicationConfig.getInstance().isShutdownRequested())
			{
				LOG.info("Abandoning startup due to shutdown request");
				return p;
			}
			
			//If we came out of the loop, because we waited the max....
			final long now = System.currentTimeMillis();
			if (now > loopUntil)
			{
				LOG.debug("Exited file-wait loop because wait time (" + now + ") exceeded maximum (" + loopUntil + ")");

				String savedPropTimeVal = PROPS.getProperty("saved.epoch_time_seconds", "0");
				
				if (currPropTimeVal.equals(savedPropTimeVal))
				{
					// We have an old copy of the file, so use it
					LOG.info("The current Prisme config file appears valid, using file: " + file.getAbsolutePath());
				}
				else
				{
					// We have an old or non-existent file
					String msg = "Error! The Prisme database YAML file is"
							+ " too old, different, or a timeout was exceeded"
							+ " while waiting for a new file: " + file.getAbsolutePath()
							+ "\n"
							+ "Attempting to fall back to non-Production mode.";
					LOG.error(msg);
				}
			} else {
				LOG.debug("Exited file-wait loop before wait time (" + now + ") exceeded maximum (" + loopUntil + "), probably because file found");
			}
		}
		catch (FileNotFoundException fnfe)
		{
			// The file exists but still not found? 
			String msg = "Error! No Prisme database YAML file found,"
					+ " or is inaccessible: " + file.getAbsolutePath()
					+ "\n"
					+ "Attempting to fall back to non-Production mode.";
			LOG.error(msg, fnfe);
		} 
		catch (IOException ioe) 
		{
			String msg = "Error! The Prisme database YAML file"
					+ " is inaccessible: " + file.getAbsolutePath()
					+ "\n"
					+ "Attempting to fall back to non-Production mode.";
			LOG.error(msg, ioe);
		} 
		finally
		{
			try 
			{
				if (stream != null)
				{
					stream.close();
				}
			} 
			catch (IOException e) 
			{
				// noop
			}
		}

		return p; 
	}
	
	
	/**
	 * Ensuring we have a proper path value, or set to use the application
	 * default.
	 */
	private void configureDbPath()
	{
		File f = new File(PROPS.getProperty("vuid.data.path", ""));
		
		if (!f.exists())
		{
			LOG.warn("The application path does not exist, using the default location of: " + PROPS.getProperty("default.vuid.data.path") + ".");
			PROPS.setProperty("vuid.data.path", PROPS.getProperty("default.vuid.data.path"));
		} else {
			LOG.debug("Found application path " + f.getAbsolutePath());
		}
	}
	
	
	/**
	 * Determine the type of database from the given properties file
	 * or default parameters, as needed.
	 */
	private void determineDatabaseType()
	{
		databaseType_ = DatabaseType.H2;
		useRealVuids_ = false;
		
		if (PROPS.containsKey("adapter"))
		{
			String jdbc = PROPS.getProperty("adapter", "h2");
			if (jdbc.toLowerCase().contains("oracle"))
			{
				databaseType_ = DatabaseType.ORACLE;
				useRealVuids_ = Boolean.valueOf(PROPS.getProperty("real_vuids", "false"));
				LOG.debug("PROPS \"adapter\" property contains \"oracle\".  Using Oracle " + (useRealVuids_ ? "with" : "without") + " real vuids");
			} else {
				LOG.debug("PROPS \"adapter\" property does not contain \"oracle\".  Using H2...");
			}
		} else {
			LOG.debug("PROPS properties does not contain \"adapter\" property");
		}
	}
	
	
	/* (non-Javadoc)
	 * @see gov.vha.vuid.rest.data.VuidService#isProductionMode()
	 */
	@Override
	public boolean isProductionMode()
	{
		return useRealVuids_;
	}
	
	
	/**
	 * Gets the type of database in use.
	 * 
	 * @return The database type name
	 */
	public String getDatabaseType()
	{
		return this.databaseType_.toString();
	}
	
	
	/**
	 * Sets up the connection to the specific database used. This can
	 * be called any time a connection is needed, if the current one is
	 * not valid.
	 */
	private void setup()
	{
		try
		{
			prepared_ = false;
			
			LOG.info("Connecting to " + databaseType_.toString() + ".");
			ApplicationConfig.getInstance().setStatus("Connecting to " + databaseType_.toString() + ".");
			
			// DB Connection assigned to dbConnection_ in respective methods
			
			switch(databaseType_)
			{
			case H2:
				final File dbFile = new File(
						PROPS.getProperty("vuid.data.path", 
								PROPS.getProperty("default.vuid.data.path")), 
						PROPS.getProperty("vuid.db.name", 
								PROPS.getProperty("default.vuid.db.name")));
				// This will fall back to an in-memory H2 version if there are any file access issues
				this.createOrOpenDatabase(dbFile);
				break;
			case ORACLE:
				if (!connectToOracle())
				{
					// If we are supposed to connect to Oracle but cannot,
					// fail and exit
					throw new RuntimeException("Unrecoverable error!"
							+ " Oracle connection failure. See log file for details.");
				}
				break;
			default:
				// H2_MEM
				this.createOrOpenDatabase(null);
				break;
			}
			
			prepared_ = true;
		}
		catch (Exception e)
		{
			String msg = "Error setting up database connection.";
			LOG.error(msg, e);
		}
	}
	
	@PreDestroy
	public void shutdown()
	{
		try
		{
			synchronized (LOG)
			{
				LOG.notifyAll();
			}
			prepared_ = false;
			Connection c = dbConnection_;
			if (c != null)
			{
				c.close();
			}
			
			// Blantantly copied from Spring Boot GitHub issues/example
			ClassLoader cl = Thread.currentThread().getContextClassLoader();
            Enumeration<Driver> drivers = DriverManager.getDrivers();
            while (drivers.hasMoreElements()) {
            	Driver driver = drivers.nextElement();
            	// Comparing instances, not values, incase Tomcat has other
            	// Drivers loaded
                if (driver.getClass().getClassLoader() == cl) {
                    LOG.info("Deregistering JDBC driver {}", driver);
                    DriverManager.deregisterDriver(driver);
                    LOG.info("JDBC driver {} deregistered.", driver);
                } else {
                	LOG.trace("Not deregistering JDBC driver {} as it does not belong to this webapp's ClassLoader", driver);
                }
            }
		}
		catch (SQLException sqle)
		{
			LOG.error("Error closing database connection or deregistering JDBC driver.", sqle);
		}
		finally
		{
			prepared_ = false;
			dbConnection_ = null;
			ApplicationConfig.getInstance().setStatus("Shutdown.");
		}
	}
	
	
	/**
	 * Uses the JDBC isValid(x) method to check the connection validity.
	 * 
	 * @return true is the connection is valid, false otherwise
	 * @throws SQLException
	 */
	private boolean isConnectionValid() throws SQLException
	{
		if (dbConnection_ != null)
		{
			return dbConnection_.isValid(5);
		}
		
		return false;
	}
	
	
	/* (non-Javadoc)
	 * @see gov.vha.vuid.rest.data.VuidService#isReady()
	 */
	@Override
	public boolean isReady()
	{
		try
		{
			return prepared_ && isConnectionValid() && testDatabase();
		}
		catch (SQLException e)
		{
			String msg = "Error validating database connection.";
			LOG.error(msg, e);
		}
		
		return false;
	}
	
	
	/**
	 * A more conclusive test, fetching a result.
	 * 
	 * @return true if row returned, false if no rows found
	 */
	private boolean testDatabase()
	{
		if (!prepared_)
		{
			LOG.warn("The database is not prepared for operation.");
			return false;
		}
		
		try {
			ResultSet rs = dbConnection_.createStatement().executeQuery("SELECT count(NEXT_VUID) FROM VUIDS");
			return rs.next();
		}
		catch (SQLException sqle)
		{
			String msg = "Error testing database.";
			LOG.error(msg, sqle);
		}
		
		return false;
	}
	
	
	/**
	 * Finds the next VUID value from an H2 database.
	 * 
	 * @return The next VUID value
	 * @throws SQLException
	 */
	private int getNextVuid() throws SQLException
	{
		Statement s = dbConnection_.createStatement();
		ResultSet rs = s.executeQuery("SELECT MAX(ABS(next_vuid)) FROM vuids");
		if (rs.next())
		{
			int nextVuid = rs.getInt(1);
			LOG.debug("Found " + nextVuid + " as the next VUID value.");
			return nextVuid;
		}
	
		return 0;
	}
	
	
	/**
	 * A method to find the next VUID from H2 and insert a new row based
	 * on the input parameters.
	 * 
	 * @param blocksize The number of VUIDs requested
	 * @param username The username of the requester
	 * @param reason The reason given for the request
	 * @return A RestVuidBlockData object with the result values from the H2 insert
	 * @throws SQLException
	 */
	private synchronized RestVuidBlockData insertH2Row(int blocksize, String reason, String username)
	throws SQLException
	{
		final String insertSql = "INSERT INTO VUIDS (next_vuid, start_vuid, end_vuid, request_datetime, request_reason, username) VALUES (?, ?, ?, now(), ?, ?)";
		
		// Get H2 max vuid, abs()
		int next = getNextVuid();
		int first = 0;
		int last = 0;
		if (next > 0)
		{
			// Create statement to write new max (negative)
			first = next * -1;
			last = (next + blocksize - 1) * -1; // Inclusive
			next = (next + blocksize) * -1;
			
			PreparedStatement ps = dbConnection_.prepareStatement(insertSql);
			ps.setInt(1,	next);
			ps.setInt(2,	first);
			ps.setInt(3,	last);
			ps.setString(4, reason);
			ps.setString(5, username);
			ps.executeUpdate();
			
			LOG.debug("Returning H2 VUID values: first=" + first
					+ ", last=" + last + ", next=" + next);
		}
		
		return new RestVuidBlockData(first, last);
	}
			
			
	/**
	 * A method to call the Oracle stored procedure PROC_REQUEST_VUID and
	 * return the out parameter values.
	 * 
	 * @param blocksize The number of VUIDs requested
	 * @param username The username of the requester
	 * @param reason The reason given for the request
	 * @return A RestVuidBlockData object with the result values from the Oracle stored procedure
	 * @throws SQLException
	 */
	private RestVuidBlockData callOracleStoredProcedure(int blocksize, String username, String reason)
	throws SQLException
	{
		final String storedProcSql = "{call PROC_REQUEST_VUID(?, ?, ?, ?, ?, ?, ?)}";
		
		// The stored procedure uses negative blocksizevalues for 
		// test/non-Production VUIDs and positive blocksize values 
		// for real VUIDs
		int bs = blocksize;
		if (!useRealVuids_)
		{
			bs = blocksize * -1;
		}
		
		CallableStatement cs = dbConnection_.prepareCall(storedProcSql);
		cs.setInt(1, bs); // in_RANGE (NUMBER)
		cs.setString(2, username); // in_USERNAME (VARCHAR2)
		cs.setString(3, reason); // in_REASON (VARCHAR2)
		cs.registerOutParameter(4, Types.INTEGER); // out_LAST_ID (NUMBER) -> next_vuid
		cs.registerOutParameter(5, Types.INTEGER); // out_START_VUID (NUMBER)
		cs.registerOutParameter(6, Types.INTEGER); // out_END_VUID (NUMBER)
		cs.registerOutParameter(7, Types.DATE); // out_REQUEST_DATETIME (DATE)

		cs.executeUpdate();

		int next = cs.getInt(4);
		int first = cs.getInt(5);
		int last = cs.getInt(6);
		// Don't care about this, currently
		//Date date = cs.getInt(7);
		
		LOG.debug("Returning Oracle VUID values: first=" + first
				+ ", last=" + last + ", next=" + next);
		
		return new RestVuidBlockData(first, last);
	}
	
	/**
	 * The method to fetch the VUID(s) from the database and store for
	 * retrieving when asked.
	 * 
	 * @param blocksize The number of VUIDs requested
	 * @param username The username of the requester
	 * @param reason The reason given for the request
	 * @return A RestVuidBlockData object with the start and end values
	 * @throws RuntimeException upon database access issues
	 */
	private RestVuidBlockData queryForVuids(int blocksize, String username, String reason) throws RuntimeException
	{
		
		RestVuidBlockData rb;
		
		try
		{
			if (!isConnectionValid())
			{
				throw new RuntimeException("Database not connected");
			}
			
			switch(databaseType_)
			{
			case H2_MEM:
			case H2:
				rb = insertH2Row(blocksize, username, reason);
				break;
			case ORACLE:
				// Call Oracle stored procedure
				rb = callOracleStoredProcedure(blocksize, username, reason);
				break;
			default:
				String msg = "Error with database type - type not determined for VUID allocation.";
				LOG.error(msg);
				throw new RuntimeException("Unrecoverable error! " + msg);
			}
			
		}
		catch (SQLException sqle)
		{
			String msg = "Problem calling stored procedure or inserting into database.";
			LOG.error(msg, sqle);
			throw new RuntimeException(msg, sqle);
		}
		
		return rb;
	}
	
	
	/**
	 * Convenience method for requesting a VUID/block of VUIDs. This
	 * logic can be easily moved into the calling class, if desired.
	 * 
	 * @param blocksize The number of VUIDs requested
	 * @param username The username of the requester
	 * @param reason The reason given for the request
	 * @return A RestVuidBlockData object with the start and end values
	 * @throws RestException if something about the request is invalid.
	 * @throws RuntimeException if an error happens accessing the DB
	 */
	@Override
	public RestVuidBlockData requestVuids(int blocksize, String username, String reason) throws RestException
	{
		LOG.info("vuid request by {} for {} vuids with reason '{}'", username, blocksize, reason);
		
		// This is set via the CONFIG_RESOURCE or custom properties file
		int maxVuidBlocksize = Integer.parseInt(
				PROPS.getProperty("vuid.request.max_blocksize", PROPS.getProperty("default.vuid.request.max_blocksize")));
		
		if (blocksize < 1)
		{
			String msg = "An invalid blocksize of " + blocksize + " was requested.";
			LOG.info(msg);
			throw new RestException(msg);
		}
		else if (blocksize > maxVuidBlocksize)
		{
			// Randy's respond on email thread asking what the current VUID request limit is
			//String msg = "The requested number of VUIDs exceeds the maximum number allowed"
					//+ " of " + NumberFormat.getInstance().format(maxVuidBlocksize_) + ". Please reduce your requested number.";
			String msg = "The requested number of VUIDs exceeds the maximum number allowed"
					+ " (" + NumberFormat.getInstance().format(maxVuidBlocksize) + ").";
			LOG.info(msg);
			throw new RestException(msg);
		}
		
		return this.queryForVuids(blocksize, username, reason);
	}

	
	/**
	 * This used the ISAAC H2DatabaseHandle class as a template. If a file 
	 * is provided, attempt to open that database. If the database file 
	 * doesn't exist, attempt to copy the included default H2 database
	 * to the given file location.  If file is null, an in-memory db is 
	 * created. Returns false if the database already existed, true if it 
	 * was newly created (either by file or in memory).
	 * 
	 * Updated to use MVSTORE format 
	 * 
	 * @param dbFile the File object representing the H2 database file to use
	 * @return false if the database already existed, true if it was newly created
	 * @throws ClassNotFoundException
	 * @throws SQLException
	 */
	private boolean createOrOpenDatabase(File dbFile) throws ClassNotFoundException
	{
		LOG.info("Attempting to create or open H2 database file: " + (dbFile != null ? dbFile.getAbsolutePath() : null) + "...");

		boolean createdNew = true;
		boolean fallbackToH2Mem = false;
		databaseType_ = DatabaseType.H2;
		
		Class.forName("org.h2.Driver");
		
		// TODO
		String h2User = "sa";
		String h2Pass = "";
		
		if (dbFile != null)
		{
			String dbFileName = dbFile.getName() + ".mv.db";
			File temp = new File(dbFile.getParentFile(), dbFileName);
			if (temp.exists())
			{
				try
				{
					LOG.info("Attempting to open existing H2 database file: " + temp.getAbsolutePath());
					dbConnection_ = DriverManager.getConnection("jdbc:h2:" + dbFile.getAbsolutePath() +";LOG=0;CACHE_SIZE=1024000;LOCK_MODE=0;;MV_STORE=TRUE",
							h2User, h2Pass);
					createdNew = false;
					LOG.info("Connected to existing H2 database file successfully.");
				}
				catch (SQLException sqle)
				{
					// Fallback to H2_MEM
					fallbackToH2Mem = true;
					String msg = "Error connecting to given H2 database file: " + dbFileName
							+ ". Attempting to fall back to in-memory H2 database.";
					LOG.error(msg, sqle);
				}
			}
			else
			{
				// Try to copy new DB file over
				String h2Resource = "/" + dbFileName;
				try
				{
					LOG.info("Attempting to copy resource H2 database file: " + h2Resource
							+ " to "+ temp.getAbsolutePath());
					File source = new File(VuidServiceImpl.class.getResource(h2Resource).toURI());
					Files.copy(source.toPath(), temp.toPath());
					dbConnection_ = DriverManager.getConnection("jdbc:h2:" + dbFile.getAbsolutePath() +";LOG=0;CACHE_SIZE=1024000;LOCK_MODE=0;;MV_STORE=TRUE",
							h2User, h2Pass);
					createdNew = true;
					LOG.info("Connected to copied resource H2 database file successfully.");
				}
				catch (IOException ioe)
				{
					// Fallback to H2_MEM
					fallbackToH2Mem = true;
					String msg = "Error copying H2 database resource file: " + h2Resource
							+ ". Attempting to fall back to in-memory H2 database.";
					LOG.error(msg, ioe);
				}
				catch (SQLException sqle)
				{
					// Fallback to H2_MEM
					fallbackToH2Mem = true;
					String msg = "Error connecting to included resource H2 database file: " + temp.getAbsolutePath()
							+ ". Attempting to fall back to in-memory H2 database.";
					LOG.error(msg, sqle);
				} 
				catch (URISyntaxException urise) 
				{
					// Fallback to H2_MEM
					fallbackToH2Mem = true;
					String msg = "Error creating URI for H2 database resource file: " + h2Resource
							+ ". Attempting to fall back to in-memory H2 database.";
					LOG.error(msg, urise);
				}
				catch (NullPointerException npe) 
				{
					// Fallback to H2_MEM
					// This will mainly show up during testing
					fallbackToH2Mem = true;
					String msg = "Error copying, creating or connecting to H2 database file: " + h2Resource
							+ ". Attempting to fall back to in-memory H2 database."
							+ System.getProperty("line.separator") 
							+ "Error of type " + npe.getClass().getSimpleName() + ": " + npe.getMessage();
					LOG.error(msg); 
				}
			}
		}
		else
		{
			LOG.info("Passed H2 database file is null");
			fallbackToH2Mem = true;
		}
		
		if (fallbackToH2Mem)
		{
			try
			{
				LOG.info("Attempting to fall back to in-memory database.");
				dbConnection_ = DriverManager.getConnection("jdbc:h2:mem:;MV_STORE=TRUE");
				initializeVuidsTable();
				createdNew = true;
				databaseType_ = DatabaseType.H2_MEM;
				LOG.info("Connected to in-memory H2 database successfully.");
			}
			catch (SQLException sqle)
			{
				databaseType_ = DatabaseType.NONE;
				String msg = "Error creating or initializing in-memory H2 database. Halting.";
				LOG.error(msg, sqle);
				throw new RuntimeException("Unrecoverable error! " + msg, sqle);
			}
		}
		
		return createdNew;
	}
	
	
	/**
	 * This attempts to provision the VUIDS table in a freshly created,
	 * in-memory H2 database. It also will load an initial row (preload)
	 * so the value '0' isn't returned.
	 * 
	 * @throws SQLException
	 */
	private void initializeVuidsTable() throws SQLException
	{
		String script = PROPS.getProperty("vuid.db.initsql", "vuids.sql");
		
		LOG.info("Attempting to preload H2 database from: " + script);
		
		String sqlProvision = "RUNSCRIPT FROM 'classpath:"
				+ script
				+ "' CHARSET 'utf-8'";
		Statement s = dbConnection_.createStatement();
		s.executeUpdate(sqlProvision);
		
		LOG.info("H2 database VUIDS table initialized.");
	}
	
	
	/**
	 * Attempts to connect to Oracle using the given connection parameters.
	 * 
	 * @return true if the connection is successful, false otherwise
	 */
	private boolean connectToOracle()
	{
		try
		{
			Class.forName("oracle.jdbc.driver.OracleDriver");
			dbConnection_ = DriverManager.getConnection(
					PROPS.getProperty("url", "no-url"), 
					PROPS.getProperty("username", "no-username"), 
					PROPS.getProperty("password", "no-password"));
			LOG.info("Connected to Oracle successfully.");
			return true;
		}
		catch (Exception e)
		{
			String msg = "Error creating Oracle database connection.";
			LOG.error(msg, e);
		}
		
		return false;
	}

	/**
	 * Returns the Prisme security_token from the database configuration file.
	 * 
	 * @return The complete log events URL, including the security token, if found
	 */
	public Optional<String> getLogEventsTargetURL()
	{
		// Returning an Optional as it's entirely possible we won't have this
		// value to return
		String logEventsUrl = PROPS.getProperty("log_events_url");
		if (StringUtils.isNotBlank(logEventsUrl))
		{
			return Optional.of(logEventsUrl);
		}
		else
		{
			return Optional.empty();
		}
	}
	
	
	/**
	 * The database password should not be persisted in any way.
	 * 
	 * @return A Properties object without the 'password' key
	 */
	private Properties getSanitizedProperties(Properties p_source)
	{
		// This is obviously a simplified method, as we're only
		// worried about very specific entries that we control
		Properties p = new Properties();
		for (Enumeration<?> e = p_source.propertyNames(); e.hasMoreElements(); ) 
		{
			String name = (String) e.nextElement();
			String value = p_source.getProperty(name);
			if (!StringUtils.containsIgnoreCase(name, "password"))
			{
				p.put(name, value);				
			}
		}
		// To clean up the properties, since this was a generated YAML file
		p.remove("---");
				
		return p;
	}
	
	
	/**
	 * Convenience method to include properties with a prepended value.
	 * 
	 * @return A Properties object with keys prepeneded with '$prepend.'
	 */
	private Properties getPrependedProperties(Properties p_source, String prepend)
	{
		if (StringUtils.isBlank(prepend))
		{
			return p_source;
		}
		
		String prepender = prepend.trim();
		
		Properties p = new Properties();
		for (Enumeration<?> e = p_source.propertyNames(); e.hasMoreElements(); ) 
		{
			String name = (String) e.nextElement();
			String value = p_source.getProperty(name);
			p.put(prepender + "." + name, value);
		}
		// To clean up the properties, since this was a generated YAML file
		p.remove(prepender + ".---");
				
		return p;
	}
}
